import type { RouteLocationNormalizedLoaded, RouteRecordRaw, _RouteRecordBase } from 'vue-router';
import type { ElegantConstRoute, LastLevelRouteKey, RouteKey, RouteMap } from '@elegant-router/types';
import { useSvgIcon } from '@/hooks/common/icon';
import { $t } from '@/locales';

/**
 * Filter auth routes by roles
 *
 * @param routes Auth routes
 * @param roles Roles
 */
export function filterAuthRoutesByRoles(routes: ElegantConstRoute[], roles: string[]) {
  return routes.flatMap(route => filterAuthRouteByRoles(route, roles));
}

/**
 * Filter auth route by roles
 *
 * @param route Auth route
 * @param roles Roles
 */
function filterAuthRouteByRoles(route: ElegantConstRoute, roles: string[]): ElegantConstRoute[] {
  const routeRoles = (route.meta && route.meta.roles) || [];

  // if the route's "roles" is empty, then it is allowed to access
  const isEmptyRoles = !routeRoles.length;

  // if the user's role is included in the route's "roles", then it is allowed to access
  const hasPermission = routeRoles.some(role => roles.includes(role));

  const filterRoute = { ...route };

  if (filterRoute.children?.length) {
    filterRoute.children = filterRoute.children.flatMap(item => filterAuthRouteByRoles(item, roles));
  }

  // Exclude the route if it has no children after filtering
  if (filterRoute.children?.length === 0) {
    return [];
  }

  return hasPermission || isEmptyRoles ? [filterRoute] : [];
}

/**
 * sort route by order
 *
 * @param route route
 */
function sortRouteByOrder(route: ElegantConstRoute) {
  if (route.children?.length) {
    // Sort children by route name to simulate creation time sorting
    route.children.sort((next, prev) => {
      // 使用路由名称进行排序，通常能反映创建顺序
      return String(next.name).localeCompare(String(prev.name));
    });
    route.children.forEach(sortRouteByOrder);
  }

  return route;
}

/**
 * sort routes by order
 *
 * @param routes routes
 */
export function sortRoutesByOrder(routes: ElegantConstRoute[]) {
  // 按路由名称排序一级路由
  routes.sort((next, prev) => {
    return String(next.name).localeCompare(String(prev.name));
  });
  // 递归排序所有子路由
  routes.forEach(sortRouteByOrder);

  return routes;
}

/**
 * Get global menus by auth routes
 *
 * @param routes Auth routes
 */
export function getGlobalMenusByAuthRoutes(routes: ElegantConstRoute[]) {
  const menus: App.Global.Menu[] = [];

  // 为每个模块定义具体的二级菜单排序顺序
  const moduleMenuOrders: Record<string, string[]> = {
    // 人工监测/越界开采系统
    home: ['home_rail', 'home_police'], // 示例排序，根据实际菜单项调整
    home_rail: ['home_rail_person'],
    // 人员定位系统
    personnel: ['personnel_realtime', 'personnel_track', 'personnel_sos', 'personnel_manage'],
    personnel_manage: ['personnel_manage_visitor', 'personnel_manage_staff'],
    personnel_statistics: ['personnel_statistics_statis', 'personnel_statistics_police', 'personnel_statistics_sheet'],
    // 全方位监测监控系统
    orientation: ['orientation_monitor', 'orientation_video'], // 示例排序
    orientation_monitor: ['orientation_monitor_history'],
    // 排土场安全监测系统
    casting: ['casting_home', 'casting_data', 'casting_monitor', 'casting_polling', 'casting_police', 'casting_system'], // 示例排序
    casting_monitor: ['casting_monitor_manage'],
    casting_polling: ['casting_polling_manage', 'casting_polling_typemanage'],
    casting_police: ['casting_police_setting'],
    // 边坡在线监测系统
    monitoring: [
      'monitoring_home',
      'monitoring_data',
      'monitoring_monitor',
      'monitoring_polling',
      'monitoring_police',
      'monitoring_system'
    ], // 示例排序
    monitoring_monitor: ['monitoring_monitor_manage'],
    monitoring_polling: ['monitoring_polling_manage', 'monitoring_polling_typemanage'],
    monitoring_police: ['monitoring_polling_setting'],
    // 卡车调度系统
    truck: ['truck_safety', 'truck_car', 'truck_area', 'truck_video', 'truck_vehicle', 'truck_smart'], // 示例排序
    truck_car: [
      'truck_car_cartype',
      'truck_car_playback',
      'truck_car_message',
      'truck_car_track',
      'truck_car_type',
      'truck_car_fault',
      'truck_car_maintain'
    ],
    truck_video: ['truck_video_monitoring'],
    // AI视频监控系统
    ai: ['ai_home', 'ai_shexiang', 'ai_suanfa', 'ai_shipin', 'ai_jieguo', 'ai_tuisong', 'ai_tixing'],
 
    // 用户管理系统WT
    user: ['user_yonghu', 'user_juese', 'user_gangwei', 'user_bumen', 'user_caidan', 'user_model', 'user_log']
  };

  routes.forEach(route => {
    if (!route.meta?.hideInMenu) {
      const menu = getGlobalMenuByBaseRoute(route);

      if (route.children?.some(child => !child.meta?.hideInMenu)) {
        // 获取子菜单
        const childMenus = getGlobalMenusByAuthRoutes(route.children);

        // 获取当前模块的排序规则
        const moduleOrder = moduleMenuOrders[route.name || ''];

        if (moduleOrder && moduleOrder.length > 0) {
          // 使用模块特定的排序规则
          childMenus.sort((next, prev) => {
            const nextIndex = moduleOrder.indexOf(next.key);
            const prevIndex = moduleOrder.indexOf(prev.key);

            if (nextIndex !== -1 && prevIndex !== -1) {
              return nextIndex - prevIndex;
            }
            if (nextIndex !== -1) return -1;
            if (prevIndex !== -1) return 1;
            return String(next.key).localeCompare(String(prev.key));
          });
        } else {
          // 没有特定排序规则时，回退到按路由名称排序
          childMenus.sort((next, prev) => {
            return String(next.key).localeCompare(String(prev.key));
          });
        }

        menu.children = childMenus;
      }

      menus.push(menu);
    }
  });

  // 一级菜单排序代码保持不变
  const menuOrder = [
    'home', // 越界开采系统
    'personnel', // 人员定位系统
    'orientation', // 全方位监测监控系统
    'casting', // 排土场安全监测系统
    'monitoring', // 边坡在线监测系统
    'truck', // 卡车调度系统
    'ai', // AI监控系统
    'user' // 用户管理系统WT
  ];

  menus.sort((next, prev) => {
    const nextIndex = menuOrder.indexOf(next.key);
    const prevIndex = menuOrder.indexOf(prev.key);

    // 如果两个菜单项都在排序数组中，按数组顺序排序
    if (nextIndex !== -1 && prevIndex !== -1) {
      return nextIndex - prevIndex;
    }
    // 如果只有一个在排序数组中，在数组中的排在前面
    if (nextIndex !== -1) return -1;
    if (prevIndex !== -1) return 1;
    // 如果都不在排序数组中，按原来的名称排序
    return String(next.key).localeCompare(String(prev.key));
  });

  return menus;
}

/**
 * Update locale of global menus
 *
 * @param menus
 */
export function updateLocaleOfGlobalMenus(menus: App.Global.Menu[]) {
  const result: App.Global.Menu[] = [];

  menus.forEach(menu => {
    const { i18nKey, label, children } = menu;

    const newLabel = i18nKey ? $t(i18nKey) : label;

    const newMenu: App.Global.Menu = {
      ...menu,
      label: newLabel
    };

    if (children?.length) {
      newMenu.children = updateLocaleOfGlobalMenus(children);
    }

    result.push(newMenu);
  });

  return result;
}

/**
 * Get global menu by route
 *
 * @param route
 */
function getGlobalMenuByBaseRoute(route: RouteLocationNormalizedLoaded | ElegantConstRoute) {
  const { SvgIconVNode } = useSvgIcon();

  const { name, path } = route;
  const {
    title,
    i18nKey,
    icon = import.meta.env.VITE_MENU_ICON,
    localIcon,
    imageIcon,
    iconFontSize
  } = route.meta ?? {};

  const label = i18nKey ? $t(i18nKey) : title!;

  // 检查是否是日志管理或模型管理的子路由
  const isLogOrModelChildRoute =
    name &&
    (String(name).startsWith('user_log_') ||
      String(name).startsWith('user_model_') ||
      String(name).startsWith('home_rail_') ||
      String(name).startsWith('casting_data_') ||
      String(name).startsWith('casting_monitor_') ||
      String(name).startsWith('casting_polling_') ||
      String(name).startsWith('casting_police_') ||
      String(name).startsWith('casting_system_') ||
      String(name).startsWith('orientation_monitor_') ||
      String(name).startsWith('personnel_manage_') ||
      String(name).startsWith('personnel_statistics_') ||
      String(name).startsWith('monitoring_data_') ||
      String(name).startsWith('monitoring_monitor_') ||
      String(name).startsWith('monitoring_polling_') ||
      String(name).startsWith('monitoring_police_') ||
      String(name).startsWith('monitoring_system_') ||
      String(name).startsWith('truck_area_') ||
      String(name).startsWith('truck_car_') ||
      String(name).startsWith('truck_safety_') ||
      String(name).startsWith('truck_smart_') ||
      String(name).startsWith('truck_vehicle_') ||
      String(name).startsWith('truck_video_'));

  const menu: App.Global.Menu = {
    key: name as string,
    label,
    i18nKey,
    routeKey: name as RouteKey,
    routePath: path as RouteMap[RouteKey],
    ...(!isLogOrModelChildRoute && {
      icon: SvgIconVNode({
        icon,
        localIcon,
        imageIcon, // 传递图片图标路径
        fontSize: iconFontSize || 15
      })
    })
  };

  return menu;
}

/**
 * Get cache route names
 *
 * @param routes Vue routes (two levels)
 */
export function getCacheRouteNames(routes: RouteRecordRaw[]) {
  const cacheNames: LastLevelRouteKey[] = [];

  routes.forEach(route => {
    // only get last two level route, which has component
    route.children?.forEach(child => {
      if (child.component && child.meta?.keepAlive) {
        cacheNames.push(child.name as LastLevelRouteKey);
      }
    });
  });

  return cacheNames;
}

/**
 * Is route exist by route name
 *
 * @param routeName
 * @param routes
 */
export function isRouteExistByRouteName(routeName: RouteKey, routes: ElegantConstRoute[]) {
  return routes.some(route => recursiveGetIsRouteExistByRouteName(route, routeName));
}

/**
 * Recursive get is route exist by route name
 *
 * @param route
 * @param routeName
 */
function recursiveGetIsRouteExistByRouteName(route: ElegantConstRoute, routeName: RouteKey) {
  let isExist = route.name === routeName;

  if (isExist) {
    return true;
  }

  if (route.children && route.children.length) {
    isExist = route.children.some(item => recursiveGetIsRouteExistByRouteName(item, routeName));
  }

  return isExist;
}

/**
 * Get selected menu key path
 *
 * @param selectedKey
 * @param menus
 */
export function getSelectedMenuKeyPathByKey(selectedKey: string, menus: App.Global.Menu[]) {
  const keyPath: string[] = [];

  menus.some(menu => {
    const path = findMenuPath(selectedKey, menu);

    const find = Boolean(path?.length);

    if (find) {
      keyPath.push(...path!);
    }

    return find;
  });

  return keyPath;
}

/**
 * Find menu path
 *
 * @param targetKey Target menu key
 * @param menu Menu
 */
function findMenuPath(targetKey: string, menu: App.Global.Menu): string[] | null {
  const path: string[] = [];

  function dfs(item: App.Global.Menu): boolean {
    path.push(item.key);

    if (item.key === targetKey) {
      return true;
    }

    if (item.children) {
      for (const child of item.children) {
        if (dfs(child)) {
          return true;
        }
      }
    }

    path.pop();

    return false;
  }

  if (dfs(menu)) {
    return path;
  }

  return null;
}

/**
 * Transform menu to breadcrumb
 *
 * @param menu
 */
function transformMenuToBreadcrumb(menu: App.Global.Menu) {
  const { children, ...rest } = menu;

  const breadcrumb: App.Global.Breadcrumb = {
    ...rest
  };

  if (children?.length) {
    breadcrumb.options = children.map(transformMenuToBreadcrumb);
  }

  return breadcrumb;
}

/**
 * Get breadcrumbs by route
 *
 * @param route
 * @param menus
 */
export function getBreadcrumbsByRoute(
  route: RouteLocationNormalizedLoaded,
  menus: App.Global.Menu[]
): App.Global.Breadcrumb[] {
  const key = route.name as string;
  const activeKey = route.meta?.activeMenu;

  for (const menu of menus) {
    if (menu.key === key) {
      return [transformMenuToBreadcrumb(menu)];
    }

    if (menu.key === activeKey) {
      const ROUTE_DEGREE_SPLITTER = '_';

      const parentKey = key.split(ROUTE_DEGREE_SPLITTER).slice(0, -1).join(ROUTE_DEGREE_SPLITTER);

      const breadcrumbMenu = getGlobalMenuByBaseRoute(route);
      if (parentKey !== activeKey) {
        return [transformMenuToBreadcrumb(breadcrumbMenu)];
      }

      return [transformMenuToBreadcrumb(menu), transformMenuToBreadcrumb(breadcrumbMenu)];
    }

    if (menu.children?.length) {
      const result = getBreadcrumbsByRoute(route, menu.children);
      if (result.length > 0) {
        return [transformMenuToBreadcrumb(menu), ...result];
      }
    }
  }

  return [];
}

/**
 * Transform menu to searchMenus
 *
 * @param menus - menus
 * @param treeMap
 */
export function transformMenuToSearchMenus(menus: App.Global.Menu[], treeMap: App.Global.Menu[] = []) {
  if (menus && menus.length === 0) return [];
  return menus.reduce((acc, cur) => {
    if (!cur.children) {
      acc.push(cur);
    }
    if (cur.children && cur.children.length > 0) {
      transformMenuToSearchMenus(cur.children, treeMap);
    }
    return acc;
  }, treeMap);
}
