import { h } from 'vue';
import type { Component } from 'vue';

/**
 * Svg icon render hook
 *
 * @param SvgIcon Svg icon component
 */
export default function useSvgIconRender(SvgIcon: Component) {
  interface IconConfig {
    /** Iconify icon name */
    icon?: string;
    /** Local icon name */
    localIcon?: string;
    /** 图片图标路径 */
    imageIcon?: string;
    /** Icon color */
    color?: string;
    /** Icon size */
    fontSize?: number;
  }

  type IconStyle = Partial<Pick<CSSStyleDeclaration, 'color' | 'fontSize'>>;

  /**
   * Svg icon VNode
   *
   * @param config
   */
  const SvgIconVNode = (config: IconConfig) => {
    const { color, fontSize, icon, localIcon, imageIcon } = config;

    const style: IconStyle = {};

    if (color) {
      style.color = color;
    }
    if (fontSize) {
      style.fontSize = `${fontSize}px`;
    }

    // 新增：优先渲染图片图标
    if (imageIcon) {
      return () => h('img', {
        src: imageIcon,
        style: {
          ...style,
          width: `${fontSize || 20}px`,
          height: `${fontSize || 20}px`,
          display: 'block',
          objectFit: 'contain'
        },
        alt: 'menu-icon'
      });
    }

    if (!icon && !localIcon) {
      return undefined;
    }

    return () => h(SvgIcon, { icon, localIcon, style });
  };

  return {
    SvgIconVNode
  };
}
