import store from "@/store";

function authPermission(permission) {
	// 定义超级权限通配符（匹配所有权限）
	const all_permission = "*:*:*";
	// 安全获取 Vuex 中的权限列表，避免 store 未初始化时报错
	const permissions = store.getters && store.getters.permissions;

	// 当且仅当传入有效权限标识时进行校验
	if (permission && permission.length > 0) {
		// 使用数组的 some 方法实现逻辑或：
		// 1. 当前用户拥有超级权限，或
		// 2. 当前用户拥有指定精确权限
		return permissions.some((v) => {
			return all_permission === v || v === permission;
		});
	} else {
		// 无效权限请求直接拒绝
		return false;
	}
}

function authRole(role) {
	// 定义超级管理员角色标识
	const super_admin = "admin";
	// 安全获取 Vuex 中的角色列表，避免 store 未初始化时报错
	const roles = store.getters && store.getters.roles;

	// 当且仅当传入有效角色标识时进行校验
	if (role && role.length > 0) {
		// 使用数组的 some 方法实现逻辑或：
		// 1. 当前用户是超级管理员，或
		// 2. 当前用户拥有指定角色
		return roles.some((v) => {
			return super_admin === v || v === role;
		});
	} else {
		// 无效角色请求直接拒绝
		return false;
	}
}

export default {
	// 验证用户是否具备某权限
	hasPermi(permission) {
		return authPermission(permission);
	},
	// 验证用户是否含有指定权限，只需包含其中一个
	hasPermiOr(permissions) {
		return permissions.some((item) => {
			return authPermission(item);
		});
	},
	// 验证用户是否含有指定权限，必须全部拥有
	hasPermiAnd(permissions) {
		return permissions.every((item) => {
			return authPermission(item);
		});
	},
	// 验证用户是否具备某角色
	hasRole(role) {
		return authRole(role);
	},
	// 验证用户是否含有指定角色，只需包含其中一个
	hasRoleOr(roles) {
		return roles.some((item) => {
			return authRole(item);
		});
	},
	// 验证用户是否含有指定角色，必须全部拥有
	hasRoleAnd(roles) {
		return roles.every((item) => {
			return authRole(item);
		});
	},
};
